/************************************************************
EINGANGSRECHNUNGSDOKUMENT
*************************************************************/
CREATE TABLE eingrechdokument (
  beld_zahl_skontiert   BOOL DEFAULT FALSE,                     -- Wenn die Rechnung bezahlt wurde, haben wir dann Skonto  bekommen?
  beld_mahnstufe        SMALLINT,                               -- Derzeitige Mahnstufe?
  beld_mahndatum        DATE,                                   -- Mahnstufe wurde wann gesetzt?
  beld_pruefer          VARCHAR(10) REFERENCES llv(ll_db_usename) ON UPDATE CASCADE,  -- Mitarbeiternummer des zuständigen Mitarbeiters für Pruefung der Rechnung auf inhaltliche Korrektheit
  beld_geprueftvon      VARCHAR(10) REFERENCES llv(ll_db_usename) ON UPDATE CASCADE,  -- Mitarbeiternummer von dem, der tatsächlich geprüft hat(kann ja auch Vertreter gewesen sein)
  beld_geprueft         BOOLEAN NOT NULL DEFAULT FALSE,         -- Beleg wurde per Assistent freigegeben, nachdem Pruefung erfolgte
  beld_pruefdatum       DATE,                                   -- An dem Datum erfolgte die Prüfung.
  beld_vertragsnummer   VARCHAR(50),
  beld_buchmonth        VARCHAR(6),                             -- Buchungszeitraum Format 'MMYYYY' Bsp: 022016 für Februar 2016
  beld_esr              varchar(50) CONSTRAINT beld_esr_chk CHECK ( beld_esr ~ '^[0-9]{1,27}$' ),  -- Einzahlungsschein-Referenznummer, numerisch mit maximal 27 Ziffern
  beld_rund             numeric                                 -- Rundungsfaktor
) INHERITS (belegdokument);
--
ALTER TABLE eingrechdokument ADD PRIMARY KEY (beld_id);

-- Rechnungsnummer darf nur einmal pro Lieferant
CREATE UNIQUE INDEX xtt11224 ON eingrechdokument(beld_krzrechnung, beld_refbeleg);

-- Dokument existiert bereits.
CREATE UNIQUE INDEX xtt10849_eingrechdokument ON eingrechdokument (beld_belegtyp, beld_dokunr);

-- index for TEingRech.GetMaxMahnstufe()
CREATE INDEX ON eingrechdokument ( beld_krzrechnung, beld_mahnstufe, beld_verbucht );


CREATE OR REPLACE FUNCTION eingrechdokument__b_i__beld_rund() RETURNS TRIGGER AS $$
BEGIN

  IF new.beld_rund IS null THEN
    new.beld_rund := wa_rund FROM bewa WHERE wa_einh = new.beld_waehr;
  END IF;

  RETURN new;
END $$ LANGUAGE plpgsql;

CREATE TRIGGER eingrechdokument__b_i__beld_rund
  BEFORE INSERT
  ON eingrechdokument
  FOR EACH ROW
EXECUTE PROCEDURE eingrechdokument__b_i__beld_rund();
--


-- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
CREATE TRIGGER eingrechdokument_set_modified
  BEFORE INSERT OR UPDATE
  ON eingrechdokument
  FOR EACH ROW
 EXECUTE PROCEDURE table_modified();
--

-- Schlüssel für Standardsuche Eingangsrechnungen (Kopf), #7157
CREATE OR REPLACE FUNCTION eingrechdokument__a_iu_keywordsearch() RETURNS TRIGGER AS $$
  BEGIN
    PERFORM TSystem.kws_create_keywords(new.*);
    RETURN new;
  END $$ LANGUAGE plpgsql;

  CREATE TRIGGER eingrechdokument__a_iu_keywordsearch
    AFTER INSERT OR UPDATE
    OF beld_dokunr, beld_refbeleg, beld_vertragsnummer
    ON eingrechdokument
    FOR EACH ROW
    EXECUTE PROCEDURE eingrechdokument__a_iu_keywordsearch();
--

-- Indexe nachsetzen
CREATE INDEX eingrechdokument_beld_dokunr ON eingrechdokument(beld_dokunr);
 CREATE INDEX eingrechdokument_beld_dokunr_like ON eingrechdokument(beld_dokunr varchar_pattern_ops);
 CREATE INDEX eingrechdokument_beld_refbeleg ON eingrechdokument(beld_refbeleg);
 CREATE INDEX eingrechdokument_beld_refbeleg_like ON eingrechdokument(beld_refbeleg varchar_pattern_ops);
 CREATE INDEX eingrechdokument_insert_date ON eingrechdokument(insert_date);
 CREATE INDEX eingrechdokument_beld_definitiv ON eingrechdokument(beld_definitiv);
 CREATE INDEX eingrechdokument_beld_verbucht ON eingrechdokument(beld_verbucht);
 CREATE INDEX eingrechdokument_beld_geprueft ON eingrechdokument(beld_geprueft, beld_pruefer);
--

-- Foreign-Keys aus Belegtabelle nachsetzen
ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_krzbesteller_fkey FOREIGN KEY (beld_krzbesteller)
        REFERENCES adressen_keys (ak_krz) ON UPDATE CASCADE;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_krzrechnung_fkey FOREIGN KEY (beld_krzrechnung)
        REFERENCES adressen_keys (ak_krz) ON UPDATE CASCADE;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_krzlieferung_fkey FOREIGN KEY (beld_krzlieferung)
        REFERENCES adressen_keys (ak_krz) ON UPDATE CASCADE  ;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_versandart_fkey FOREIGN KEY (beld_versandart)
        REFERENCES versart (v_id) ON UPDATE CASCADE  ;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_waehr_fkey FOREIGN KEY (beld_waehr)
        REFERENCES bewa (wa_einh) ON UPDATE CASCADE  ;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_zahlart_fkey FOREIGN KEY (beld_zahlart)
        REFERENCES zahlart(za_id) ON UPDATE CASCADE  ;

 ALTER TABLE eingrechdokument
  ADD CONSTRAINT eingrechdokument_beld_belegart_fkey FOREIGN KEY (beld_belegart)
        REFERENCES belegart (bela_id) ON UPDATE CASCADE;




--

--Von Belegdokument geerbte Trigger erstellen.
CREATE TRIGGER register_beleg_id
 AFTER INSERT OR UPDATE OR DELETE
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE register_beleg_id();
--

--Von Belegdokument geerbt.
CREATE TRIGGER belegdokument__b_iu
 BEFORE INSERT OR UPDATE
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE belegdokument__b_iu();
--

--Von Belegdokument geerbt.
CREATE TRIGGER belegdokument__a_i
 AFTER INSERT
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE belegdokument__a_i();
--

--Von Belegdokument geerbt.
CREATE TRIGGER belegdokument__a_d
 AFTER DELETE
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE belegdokument__a_d();
--

--Von Belegdokument geerbt.
CREATE TRIGGER belegdokument__a_u
 AFTER UPDATE
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE belegdokument__a_u();
--

--
CREATE OR REPLACE FUNCTION eingrechdokument__b_u() RETURNS TRIGGER AS $$
 BEGIN
  -- Falls Zahlungslauf/DTA benutzt wird
  -- => Zahlart muss eingegeben sein, bevor Pruefer zugewiesen wird. Sonst kann Rechnung freigegeben werden und taucht im Zahlungslauf auf, obwohl noch ungeprueft
  IF TSystem.Settings__GetBool('ER_Prodat_Zahlung') AND (new.beld_zahlart IS NULL)  AND (NOT new.beld_pruefer IS NULL) THEN
    RAISE EXCEPTION 'xtt10913';
  END IF;

  -- Beim geprueft setzen automatisch das Pruefdatum nachtragen
  IF (NOT old.beld_geprueft) AND (new.beld_geprueft) AND (new.beld_pruefdatum IS NULL) THEN
    new.beld_pruefdatum:=current_date;
  END IF;

  -- Wenn Beleg vollstaendig bezahlt / als abgeschlossen betrachtet, das Datum nachtragen
  IF (NOT old.beld_verbucht) AND new.beld_verbucht AND ( new.beld_abschlussdatum IS NULL )THEN
    new.beld_abschlussdatum:=current_date;
  END IF;

  RETURN new;
 END $$ LANGUAGE plpgsql;
--
CREATE TRIGGER eingrechdokument__b_u
 BEFORE UPDATE
 ON eingrechdokument
 FOR EACH ROW
 EXECUTE PROCEDURE eingrechdokument__b_u();
--


/* Mitschreiben, wer die Rechnung geprueft gesetzt hat. */
CREATE OR REPLACE FUNCTION eingrechdokument__b_u_SetPruefer() RETURNS TRIGGER AS $$
 BEGIN
  new.beld_geprueftvon := current_user;
  RETURN new;
 END $$ LANGUAGE plpgsql;
--

--
CREATE TRIGGER eingrechdokument__b_u_SetPruefer
  BEFORE UPDATE OF beld_geprueft
  ON eingrechdokument
  FOR EACH ROW
  WHEN (new.beld_geprueft AND NOT old.beld_geprueft)
 EXECUTE PROCEDURE eingrechdokument__b_u_SetPruefer();
--

--
CREATE OR REPLACE FUNCTION TBeleg.EingrechStatus(IN beldok INTEGER) RETURNS VARCHAR AS $$
 DECLARE rec RECORD;
  result VARCHAR(40);
 BEGIN

  SELECT * INTO rec FROM eingrechdokument WHERE beld_id = beldok;

  IF Not rec.beld_geprueft THEN
      result:= lang_text(10814);   --Ungeprüft
    ELSEIF rec.beld_geprueft AND NOT (rec.beld_freigabe OR rec.beld_verbucht ) Then
      result:= lang_text(10815);   --Geprueft
    ELSEIF rec.beld_freigabe AND NOT (rec.beld_verbucht ) THEN
      result:= lang_text(10816);   --Freigegeben
    ELSEIF  rec.beld_verbucht THEN
      result:= lang_text(10817);   --Bezahlt
  END IF;

  IF rec.beld_definitiv THEN
    result:=lang_text(10935) || ' | ' || COALESCE(result,'');
  END IF;

  RETURN COALESCE(result,'');
 END $$ LANGUAGE plpgsql STABLE;
--

--Returns belegdokid
CREATE OR REPLACE FUNCTION TBeleg.EingrechDokumentFromWendat(IN wendatdbrid INTEGER, IN belegdoknr VARCHAR(30)) RETURNS INTEGER AS $$ -- TWawi.beleg_k__rechnunge__from__adk2__wendat__create
 DECLARE rec     RECORD;
         docRec  RECORD;
         frei    BOOLEAN;
         belegid INTEGER;
         docid   INTEGER;
         pruefer VARCHAR;
 BEGIN

  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;

  -- Belegdokument noch anlegen ?
  IF NOT (docid IS NULL) THEN
    RETURN docid;
  END IF;

  --Freier Wendat Satz?
  SELECT (w_lds_id IS NULL) INTO frei FROM wendat WHERE wendat.dbrid = wendatdbrid;

  -- Wir sammeln mal alle möglichen Daten im System
  SELECT IfThen(frei, a2_vers, ltd_vers) AS versart,
         w_zugang, ld_id, w_wen, w_aknr, w_l_krz, COALESCE(ld_waer, TSystem.Settings__Get('BASIS_W')) AS waer, w_lfsnr, ltd_gesrab,ld_eklos,
         ld_an_nr, ld_lkontakt, ld_lkontaktkrzl, ld_kontakt, ld_krzl, ld_krzf, ltd_zakbem,
         ltd_apkrzl, ltd_ap, ltd_apint, a2_zahlart,
         ldt_txt,
         COALESCE(ltd_zak, a2_zak) AS zahlkond,
         COALESCE(ltd_sks, a2_sks) AS skontosatz,
         COALESCE(ltd_skv, a2_skv) AS skontotage
      INTO rec
         FROM wendat    JOIN adk ON ad_krz = w_l_krz
                        JOIN adk2 ON ad_krz = a2_krz
                        LEFT JOIN ldsdok ON w_lds_id = ld_id
                        LEFT JOIN ldsdokdokutxt ON ld_dokunr = ltd_dokunr
                        LEFT JOIN ldsdoktxt     ON ldt_code = ld_code AND ldt_auftg = ld_auftg -- Zusatztext zur Bestellung kommt daher.
         WHERE wendat.dbrid = wendatdbrid;


  pruefer := NULL;
  -- Besteller soll Rechnungspruefer werden
  IF TSystem.Settings__GetBool('ERGPrueferIsEK') AND (COALESCE(rec.ld_kontakt,'') <> '')   THEN
    -- Wenn Zahlungslauf aktiv ist, darf das aber nur passieren, wenn die Zahlungsart feststeht.
    IF TSystem.Settings__GetBool('ER_Prodat_Zahlung') AND (rec.a2_zahlart IS NULL) THEN
      pruefer:=NULL;
    ELSE
      pruefer := rec.ld_kontakt;
    END IF;
  END IF;

  -- Belegdokument anlegen
  INSERT INTO eingrechdokument (
      beld_belegtyp, beld_dokunr, beld_apkrzl, beld_ap, beld_apint, beld_zak, beld_skv, beld_sks, beld_versandartbem,
      beld_zakbem, beld_gesamtrabatt, beld_krzbesteller, beld_krzlieferung, beld_krzrechnung, beld_waehr,
      beld_zahlart, beld_pruefer, beld_kopftext
  ) VALUES (
      'ERG', belegdoknr, rec.ltd_apkrzl, rec.ltd_ap, rec.ltd_apint, rec.zahlkond, rec.skontotage, rec.skontosatz, rec.versart,
      rec.ltd_zakbem, COALESCE(rec.ltd_gesrab,0), COALESCE(rec.ld_krzl,'#'), rec.w_l_krz, rec.w_l_krz, rec.waer,
      rec.a2_zahlart, pruefer, rec.ldt_txt
  );

  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;

  RETURN docid;
 END $$ LANGUAGE plpgsql VOLATILE;
--

--Returns belegdokid
CREATE OR REPLACE FUNCTION TBeleg.EingrechDokumentFromLdsdok(IN ldsdokdbrid INTEGER, IN belegdoknr VARCHAR(30)) RETURNS INTEGER AS $$ -- TWawi.beleg_k__rechnunge__from__adk2__einkauf__create
 DECLARE rec     RECORD;
         docRec  RECORD;
         belegid INTEGER;
         docid   INTEGER;
         pruefer VARCHAR;
 BEGIN

  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;

  -- Belegdokument noch anlegen ?
  IF NOT (docid IS NULL) THEN
    RETURN docid;
  END IF;

  -- Wir sammeln mal alle möglichen Daten im System
  SELECT ltd_vers AS versart,
         ld_stk, ld_id, ld_aknr, ld_kn, ld_waer, ld_dokunr, ltd_gesrab,ld_eklos,
         ld_an_nr, ld_lkontakt, ld_lkontaktkrzl, ld_kontakt, ld_krzl, ld_krzf, ltd_zakbem,
         ltd_apkrzl, ltd_ap, ltd_apint, a2_zahlart,
         COALESCE(ltd_zak, a2_zak) AS zahlkond,
         COALESCE(ltd_sks, a2_sks) AS skontosatz,
         COALESCE(ltd_skv, a2_skv) AS skontotage,
         ldt_txt
      INTO rec
         FROM ldsdok    JOIN adk ON ad_krz = ld_kn
                        LEFT JOIN adk2 ON ad_krz = a2_krz
                        LEFT JOIN ldsdokdokutxt ON ld_dokunr = ltd_dokunr
                        LEFT JOIN ldsdoktxt     ON ldt_code = ld_code AND ldt_auftg = ld_auftg -- Zusatztext zur Bestellung kommt daher.
         WHERE ldsdok.dbrid = ldsdokdbrid;

  pruefer := NULL;
  -- Besteller soll Rechnungspruefer werden
  IF TSystem.Settings__GetBool('ERGPrueferIsEK') AND (COALESCE(rec.ld_kontakt,'') <> '')   THEN
    -- Wenn Zahlungslauf aktiv ist, darf das aber nur passieren, wenn die Zahlungsart feststeht.
    IF TSystem.Settings__GetBool('ER_Prodat_Zahlung') AND (rec.a2_zahlart IS NULL) THEN
      pruefer:=NULL;
    ELSE
      pruefer := rec.ld_kontakt;
    END IF;
  END IF;

  -- Belegdokument anlegen
  --   Falls Zahlungslauf bzw. DTA/SEPA aktiv ist, ist Zahlart Pflicht bevor Pruefer zugewiesen wird.
  INSERT INTO eingrechdokument (
      beld_belegtyp, beld_dokunr, beld_apkrzl, beld_ap, beld_apint, beld_zak, beld_skv, beld_sks, beld_versandartbem,
      beld_zakbem, beld_gesamtrabatt, beld_krzbesteller, beld_krzlieferung, beld_krzrechnung, beld_waehr, beld_zahlart,
      beld_pruefer, beld_kopftext
  ) VALUES (
      'ERG', belegdoknr, rec.ltd_apkrzl, rec.ltd_ap, rec.ltd_apint, rec.zahlkond, rec.skontotage, rec.skontosatz, rec.versart,
      rec.ltd_zakbem, COALESCE(rec.ltd_gesrab,0), COALESCE(rec.ld_krzl,'#'), rec.ld_kn, rec.ld_kn, rec.ld_waer, rec.a2_zahlart,
      pruefer, rec.ldt_txt
  );

  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;

  RETURN docid;
 END $$ LANGUAGE plpgsql VOLATILE;
--

--Eingangsrechnung erstellen auf Kreditorendaten, Returns belegdokid
CREATE OR REPLACE FUNCTION TBeleg.EingrechDokumentFromADK2(IN adkrz VARCHAR, IN belegdoknr VARCHAR(30)) RETURNS INTEGER AS $$ -- TWawi.beleg_k__rechnunge__from__adk2__create
 DECLARE rec     RECORD;
         docRec  RECORD;
         belegid INTEGER;
         docid   INTEGER;

 BEGIN
  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;
  -- Belegdokument noch anlegen ?
  IF (docid IS NOT NULL) THEN
    RETURN docid;
  END IF;

  SELECT a2_vers AS versart,
         a2_knr, a2_krz, a2_waco, a2_rabatt,
         ad_krz,
         a2_zahlart,
         a2_zak AS zahlkond,
         a2_sks AS skontosatz,
         a2_skv AS skontotage
      INTO rec
         FROM adk2
            JOIN adk ON ad_krz = a2_krz
         WHERE a2_krz = adkrz;

  -- Belegdokument anlegen
  INSERT INTO eingrechdokument (
      beld_belegtyp, beld_dokunr, beld_apint, beld_zak, beld_skv, beld_sks, beld_versandartbem, beld_gesamtrabatt,
      beld_krzbesteller, beld_krzlieferung, beld_krzrechnung, beld_waehr, beld_zahlart, beld_erstelldatum
  ) VALUES (
      'ERG', belegdoknr, current_user, rec.zahlkond, rec.skontotage, rec.skontosatz, rec.versart, COALESCE(rec.a2_rabatt,0),
      '#', rec.ad_krz, rec.ad_krz, rec.a2_waco, rec.a2_zahlart, current_date
  );

  SELECT beld_id INTO docid FROM eingrechdokument WHERE beld_dokunr = belegdoknr;

  RETURN docid;
 END $$ LANGUAGE plpgsql VOLATILE;
--

/************************************************************
EINGANGSRECHNUNGSZAHLUNGEN
*************************************************************/

/************************************************************
ZAHLUNGSLÄUFE / SEPA / Eingangsrechnungen verbuchen, Alte DTA - Datenstrukturen
*************************************************************/

-- Zahlungslauftyp ( Inlandsueberweisung per DTA, Manuelle Ueberweisung, Auslandsrechnungen etc....)
CREATE TABLE zahlungslauftyp (
  zlt_id                SERIAL PRIMARY KEY,
  zlt_bez               VARCHAR(100),
  zlt_default           BOOLEAN NOT NULL DEFAULT FALSE -- Default Vorschlag
 );
--

-- Fasst (Eingangs-) Rechnungen zu Zahlungsläufen zusammen.
CREATE TABLE zahlungslauf (
  zl_nr                VARCHAR(40) PRIMARY KEY,        -- Zahlungslauf
  zl_typ               INTEGER NOT NULL DEFAULT 0 REFERENCES zahlungslauftyp ON UPDATE CASCADE,        -- Zahlungslauftyp ( Inlandsueberweisung per DTA, Manuelle Ueberweisung, Auslandsrechnungen etc....)
  zl_faelligbis        DATE NOT NULL,                  -- Alle Rechnungen die bis da faellig werden
  zl_vorfaelligkeit    INTEGER NOT NULL DEFAULT 0,     -- Reserve in Tagen wg. Dauer von Ueberweisungen etc
  zl_skontobis         DATE,                           -- Rechnungen die bis da Skonto haben
  zl_vorskonto         INTEGER NOT NULL DEFAULT 0,     -- Reserve in Tagen wg. Dauer von Ueberweisungen etc
  zl_adkrz             VARCHAR(19) NOT NULL DEFAULT '%',  -- Adresskürzel, absichtlich keine Referenz, damit % und sowas da rein kann.
  zl_planzahldat       DATE NOT NULL,                  -- An dem Tag soll das Zeug bezahlt werden. Braucht man für Skontoberechnungen.
  zl_txt               TEXT,                           -- Zusatztext mit irgendwelchen Kommentaren
  zl_gebucht           BOOLEAN NOT NULL DEFAULT FALSE, -- Zahlungslauf erledigt / alles bezahlt
  zl_vorbereitet       BOOLEAN NOT NULL DEFAULT FALSE, -- Rechnungen und Beträge in Zahlungslauf festgesetzt und Überweisungsvorlagen erstellt. Kein Editieren mehr in Positionen,nur Anpassung von Überweisungen.
  zl_nurgeprueft       BOOLEAN NOT NULL,               -- Zahlungslauf wurde mit der Option nur gepruefte Rechnungen erstellt.
  zl_konto             VARCHAR(100) REFERENCES ktovz ON UPDATE CASCADE  --Zahlungen werden von diesem Konto bedient.
  );
--

-- Eingangsrechnung wieder für die Bezahlung öffnen, wenn Teilzahlungen gelöscht werden.
CREATE OR REPLACE FUNCTION zahlungslauf__a_d() RETURNS TRIGGER AS $$
 BEGIN
  UPDATE eingrechdokument SET
    beld_freigabe = FALSE,
    beld_verbucht = FALSE,
    beld_zahl_skontiert = FALSE
   WHERE EXISTS(SELECT true FROM eingrechzahlung WHERE belzlg_zl_nr = old.zl_nr AND belzlg_dokument_id = beld_id);
  RETURN old;
 END $$ LANGUAGE plpgsql;

 CREATE TRIGGER zahlungslauf__a_d
 AFTER DELETE
 ON zahlungslauf
 FOR EACH ROW
 EXECUTE PROCEDURE zahlungslauf__a_d();
--

-- Zahlungen auf einen Beleg (Eingangsrechnung) oder ein Dokument (Ausgangsrechnung) erfolgen.
CREATE TABLE eingrechzahlung (
  belzlg_zl_nr          VARCHAR(40) REFERENCES zahlungslauf ON UPDATE CASCADE ON DELETE CASCADE,
  -- [OBSOLET] Altes DTA Format. Seit 08/2014 belzlg_dta_id         INTEGER, -- XTableConstraints REFERENCES dtaus DEFERRABLE, --Falls wir das per DTA bezahlen steht hier, mit welchem Überweisungssatz. Mehrere Teilzahlungen können gleiche DTA haben.
  belzlg_sepa_id        INTEGER, -- XTableConstraints REFERENCES sepaTransaktion ON UPDATE CASCADE ON DELETE SET NULL
  belzlg_belzlg_id      INTEGER
) INHERITS (belegzahlung);
--

-- Keys + Trigger nachsetzen
ALTER TABLE eingrechzahlung ADD PRIMARY KEY (belzlg_id);
ALTER TABLE eingrechzahlung ADD FOREIGN KEY (belzlg_belzlg_id) REFERENCES eingrechzahlung ON UPDATE CASCADE ON DELETE CASCADE;
    -- Referenz auf Ausgleichszahlung oder verrechnete Gutschrift.
    -- "Verminderte" Zahlung => Referenziert "Verrechnungszahlung" => Referenziert "Zu verrechnende Gutschruft"

 ALTER TABLE eingrechzahlung
  ADD CONSTRAINT eingrechzahlung_belzlg_dokument_id_fkey FOREIGN KEY (belzlg_dokument_id)
        REFERENCES belegdokument_keys(beld_key) ON UPDATE CASCADE  ;

 ALTER TABLE eingrechzahlung
  ADD CONSTRAINT eingrechzahlung_belzlg_zahlart_fkey FOREIGN KEY (belzlg_zahlart)
        REFERENCES zahlart(za_id) ON UPDATE CASCADE  ;



 -- Trigger VORDEFFINIERT -> entsprechend automatischem DBRID-Trigger "{table}_set_modified" für table_modified()
 CREATE TRIGGER eingrechzahlung_set_modified
  BEFORE INSERT OR UPDATE
  ON eingrechzahlung
  FOR EACH ROW
  EXECUTE PROCEDURE table_modified();

 CREATE TRIGGER belegzahlung__a_iud
  AFTER INSERT OR UPDATE OR  DELETE
  ON eingrechzahlung
  FOR EACH ROW
  EXECUTE PROCEDURE belegzahlung__a_iud();
--

-- Vorbereitet Sepa-Überweisung löschen, wenn zugehörige Zahlung rausfliegt.
CREATE OR REPLACE FUNCTION eingrechzahlung__a_d() RETURNS TRIGGER AS $$
 BEGIN
  DELETE FROM sepaTransaktion WHERE sepa_id = old.belzlg_sepa_id;

  RETURN old;
 END $$ LANGUAGE plpgsql;
--
CREATE TRIGGER eingrechzahlung__a_d
 AFTER DELETE
 ON eingrechzahlung
 FOR EACH ROW
 EXECUTE PROCEDURE eingrechzahlung__a_d();
--


/************************************************************
SEPA / Eingangsrechnungen verbuchen, Alte DTA - Datenstrukturen
*************************************************************/

-- Transaktions-Sammlung von Sepaüberweisungen. 1 Ausgangskonto => N-Empfänger
CREATE TABLE sepaBatch (
  seb_id               SERIAL PRIMARY KEY           ,
  seb_msgId            VARCHAR(35) NOT NULL         ,
  seb_zl_nr            VARCHAR(40)  NOT NULL REFERENCES zahlungslauf ON UPDATE CASCADE ON DELETE CASCADE,
  seb_kto_name         VARCHAR(100) NOT NULL REFERENCES ktovz, -- Eigenes Konto, von dem überwiesen wird aus Kontenverzeichnis. -- Pro Sepa Datei darf nur ein Ausgangskonto verwendet werden.
  seb_art              INTEGER NOT NULL DEFAULT 0,
  seb_xml_txt          TEXT
  -- 0 = Überweisung, 10 = Lastschrift (Basis), 11 = Lastschrift (Firmen - B2B) => Aktuell unterstützen wir, wie beim alten DTA Format, nur Überweisungen.
 );
--

-- MessageID generieren, wenn noch keine da ist.
CREATE OR REPLACE FUNCTION sepaBatch__b_00_iu_defaults() RETURNS TRIGGER AS $$
 BEGIN
  -- Die SEPA Message ID ist eine eindeutige 35-stellige Nachrichtidentifizierung (in Verbindung mit der Kunden ID oder der SEPA IBAN).
  -- Bei uns Zahlungslaufnummer+'.'+'DDMMYYYY.HHMMSS' (Erzeugungszeitpunkt). Das erlaubt perspektivisch mehrere Batches pro Zahlungslauf,
  -- um z.Bsp. Rechnungen von verschiedenen Konten zu bedienen. Aktuell nur 1:1 Abbildung von Zahlungslauf auf SepaBatch.
  IF (COALESCE(new.seb_MsgId,'')) = '' THEN
    -- (Maximal) 16 Zeichen der Zahlungslauf-Nummer + 16 Zeichen Timestamp (inkl. Punkte)
    new.seb_MsgId := SUBSTRING(COALESCE(new.seb_zl_nr,'') FROM 0 FOR 16) || '.' || to_char(current_timestamp, 'DDMMYYYY.HH24');
  END IF;

  RETURN new;
 END$$LANGUAGE plpgsql;

 CREATE TRIGGER sepaBatch__b_00_iu_defaults
  BEFORE INSERT OR UPDATE OF seb_zl_nr
  ON sepaBatch
  FOR EACH ROW
  EXECUTE PROCEDURE sepaBatch__b_00_iu_defaults();
--

-- SEPA-Überweisungsdatensätze. Zur Zeit keine Unterstützung für Lastschriften usw.
CREATE TABLE sepaTransaktion (
  sepa_id               SERIAL PRIMARY KEY          ,
  sepa_seb_id           INTEGER NOT NULL REFERENCES sepaBatch ON UPDATE CASCADE ON DELETE CASCADE,
  -- Empfängerdaten. Analog Kreditorendaten aus Adk2, aber nochmal geführt, damit nachvollziehbar bei Änderungen in der Zukunft
  sepa_adkrz            VARCHAR(21) NOT NULL REFERENCES adk ON UPDATE CASCADE, -- Kürzel der Empfängeradresse
  sepa_name             VARCHAR(27) NOT NULL        , -- Name des Empfängerkontoinhabers
  sepa_iban             VARCHAR(34) NOT NULL        , -- IBAN des Empfänger
  sepa_bic              VARCHAR(11)                 , -- BIC des Empfängers, darf bei Inlandsüberweisung leer sein
  sepa_bank             VARCHAR(60)                 , -- Bankbezeichnung, nur hilfsweise angezeigt
  sepa_EndToEndID       VARCHAR(32)                 , -- SEPA - EndToEndID, wird (angeblich) sicher zum Empfänger durchgeleitet. Verwendungszweck darf die Bank einfach abschneiden?!
                                                      -- Hier schicken wir also die Rechnungsnummer des Lieferanten mit, damit die nicht auf einmal weg ist. http://www.hettwer-beratung.de/sepa-spezialwissen/sepa-glossar/
  sepa_vzweck           VARCHAR(140)                , -- Verwendungszweck Max. 140 Zeichen, macht wegen Teilumstellung der Banken z.Zeit Probleme (06/2014)
  sepa_betrag           NUMERIC(12,2) NOT NULL      , -- Nicht-Negativer Betrag
  sepa_termin           DATE                        , -- Ausführungsdatum, Leer = Sofort
  sepa_datum            DATE NOT NULL DEFAULT current_date, -- Belegdatum
  sepa_done             BOOLEAN NOT NULL DEFAULT FALSE -- Erledigt-Kennzeichen
  -- sepa_lart             VARCHAR(4)                  , -- Lastschriftart, zur Zeit nicht benutzt, da keine Lastschriften unterstützt // OOFF = einmalig // FRST = erste // RCUR = regelmäßige // FNAL = letzte Lastschrift
   -- sepa_mandat           // nur lastschriften, nicht unterstützt
   -- sepa_Mandataenderung, // Mandatsänderung für Lastschriften (true/false)
   -- sepa_MandatDatum,     // Datum der Unterschrift des Mandats für Lastschriften
 );
--

-- Filtert offene Rechnungen nach den Zahlungslaufkriterien, erstellt Zahlungen dafür mit Skontoberücksichtigung etc.
CREATE OR REPLACE FUNCTION TBeleg.Create_ERGZahlungen( zlnr  VARCHAR(40) ) RETURNS VARCHAR AS $$
 DECLARE r  RECORD;
         zl Zahlungslauf;
         IsFaellig BOOLEAN;
         CanSkonto BOOLEAN;
         sebid     INTEGER;
         zlgid     INTEGER;
         msg       VARCHAR;
 BEGIN
  sebid := NULL;
  msg   := '';

  -- Vorbereitete Teilzahlungen und zugehörige Überweisungen löschen
  DELETE FROM eingrechzahlung WHERE belzlg_zl_nr = zlnr;
  DELETE FROM sepaBatch WHERE seb_zl_nr = zlnr;
  -- DELETE FROM dtaus WHERE dta_zl_nr = zlnr; -- Tabelle gibt es seit SEPA Umstellung nicht mehr. In Altsystem abernoch vorhanden.
  msg:= msg || IfThen(msg <> '', E'\r\n','') || 'Zahlungslauf ' || zlnr ||': Freigaben aufgehoben, vorbereitete Teilzahlungen entfernt.';
  --
  zl := zahlungslauf FROM zahlungslauf WHERE zahlungslauf.zl_nr = zlnr;


  IF zl IS NULL THEN
    RAISE EXCEPTION 'TBeleg.Create_ERGZahlungen: %', Format(lang_text(29180) /*Zahlungslauf % wurde nicht gefunden. Vorgang abgebrochen.'*/, zlnr);
  END IF;

  FOR r IN (SELECT eingrechdokument.*, ad_landiso
              FROM eingrechdokument
                LEFT JOIN adk ON beld_krzrechnung = ad_krz
              WHERE beld_belegtyp = 'ERG'           -- Eingangsrechnungen
                AND beld_belegart = 1               -- ... nur richtige
                AND beld_zahlart  = 2               -- ... die Zahlart 'Überweisung' haben (seit Ticket 442)
                AND beld_erstelldatum IS NOT NULL   -- ... mit Erstelldatum
                AND beld_refbeleg     IS NOT NULL   -- ... und Rechnungsnummer des Lieferanten
                AND beld_krzrechnung LIKE COALESCE(zl.zl_adkrz,'%') -- ... ggf. Lieferanteneinschränkung
                AND NOT beld_verbucht               -- ... noch nicht verrechnet
                AND NOT EXISTS(SELECT true FROM eingrechzahlung WHERE belzlg_dokument_id = beld_id AND (NOT belzlg_gebucht)) -- ... oder in anderem Zahlungslauf
             ORDER BY beld_krzrechnung, beld_erstelldatum

  ) LOOP

    -- Datumsprüfung, Zahlungsdatum < Zahlungslaufbeschränlung abzgl. Vorlauf ?
    isFaellig  :=( (r.beld_erstelldatum :: DATE) + COALESCE(r.beld_zak,0) <= (zl.zl_faelligbis :: DATE ) - zl.zl_vorfaelligkeit);

    -- Skontoprüfung, Rechnung kann Skonto ziehen und Zeitraum noch nicht abgelaufen, aber schon im SkontoBis - Bereich
    CanSkonto  := (COALESCE( r.beld_skv, 0) > 0)
                 AND (zl.zl_skontobis IS NOT NULL)
                 AND (zl.zl_planzahldat <= (r.beld_erstelldatum + r.beld_skv));
                 --AND ((r.beld_erstelldatum::Date) + r.beld_skv) BETWEEN current_date AND (zl.zl_skontobis - COALESCE(zl.zl_vorskonto,0));

    -- Muss erstmal nicht bezahlt werden
    IF NOT (isFaellig OR CanSkonto)                THEN CONTINUE; END IF;

    -- Wir dürfen die nur nach Rechnungsprüfung freigeben.
    IF zl.zl_nurgeprueft AND (NOT r.beld_geprueft) THEN CONTINUE; END IF;

    --
    CASE
       WHEN (zl.zl_typ = 0) THEN CONTINUE; -- Das war mal DTA. Das machen wir nicht mehr.
       WHEN (zl.zl_typ = 1) AND ((r.beld_waehr <> 'EUR') OR (COALESCE(r.ad_landiso,'') <> 'DE')) THEN CONTINUE; -- EUR, Inland, manuell
       WHEN (zl.zl_typ = 2) AND ((r.beld_waehr <> 'EUR') OR (COALESCE(r.ad_landiso,'') =  'DE')) THEN CONTINUE; -- EUR, Ausland, manuell
       WHEN (zl.zl_typ = 3) AND (r.beld_waehr = 'EUR') THEN CONTINUE; -- Fremdwährungsrechnung
       -- WHEN (zl.zl_typ = 4) THEN TRUE; Alle rechnungen.
       WHEN (zl.zl_typ = 5) AND ((r.beld_waehr <> 'EUR') OR (COALESCE(r.ad_landiso,'') <> 'DE')) THEN CONTINUE; -- EUR, Inland, SEPA
       ELSE
    END CASE;

    -- Angekommen? Dann sollte alles passen und wir legen die Teilzahlung über den Restbetrag an.
    INSERT INTO eingrechzahlung ( belzlg_dokument_id, belzlg_betrag, belzlg_zahlart, belzlg_zl_nr, belzlg_date )
      VALUES (r.beld_id,
              TBeleg.DokumentSumme( r.beld_id,
                                    FALSE, -- In Belegwährung
                                    FALSE, -- Brutto
                                    TRUE,  -- Mit Rabatten
                                    CanSkonto, -- Mit oder ohne Skonto
                                    FALSE)  -- Über volle Menge
              - r.beld_bezahlt,     -- Abzüglich schon gezahlten Betrag
              r.beld_zahlart,
              zlnr,
              COALESCE(zl.zl_planzahldat, current_date)) -- Geplantes Zahlungsdatum weitergeben
      RETURNING belzlg_id INTO zlgid;
    --msg:= msg || IfThen(msg <> '', E'\r\n','') || 'Zahlungslauf ' || zlnr ||': Zahlung erstellt für Beleg ' || r.beld_dokunr;
    msg:= msg || IfThen(msg <> '', E'\r\n','') || Format(lang_text(29220) /*'Zahlungslauf %: Zahlung erstellt für Beleg %'*/, zlnr, r.beld_dokunr);
  END LOOP;

  UPDATE zahlungslauf SET zl_vorbereitet = FALSE WHERE zl_nr = zlnr;
  RETURN msg;
 END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

CREATE OR REPLACE FUNCTION TBeleg.Assign_Gutschrift( gsID INTEGER, zlgID INTEGER) RETURNS VARCHAR AS $$
 DECLARE gs        eingRechzahlung;  -- Gutschriftsdatensatz
         zlg       eingRechzahlung;  --
         vID       INTEGER;
         sebid     INTEGER;
         zl        RECORD;
 BEGIN
  SELECT * INTO gs  FROM eingrechzahlung WHERE belzlg_id = gsID;
  gs  :=  eingrechzahlung FROM eingrechzahlung WHERE belzlg_id = gsID;
  zlg :=  eingrechzahlung FROM eingrechzahlung WHERE belzlg_id = zlgid;
  SELECT * INTO zl FROM zahlungslauf WHERE zl_nr = gs.belzlg_zl_nr;

  -- Nur wenn noch nicht verbucht.
  IF zl.zl_gebucht OR gs.belzlg_gebucht OR zlg.belzlg_gebucht THEN
    RETURN NULL;
  END IF;

  IF (zlg.belzlg_betrag - Abs(gs.belzlg_betrag)) <= 0 THEN
    RAISE EXCEPTION 'TBeleg.Assign_Gutschrift: %', Format(lang_text(29181) /*Rechnungsbetrag ist kleiner zu verrechnender Gutschrift'*/);
  END IF;

  -- Verrechnungsteilzahlung anlegen, Referenziert auf Gutschrift
  INSERT INTO eingrechzahlung ( belzlg_dokument_id, belzlg_tan, belzlg_date, belzlg_betrag, belzlg_zahlart, belzlg_txt, belzlg_belzlg_id, belzlg_zl_nr )
    VALUES ( zlg.belzlg_dokument_id, zlg.belzlg_tan, zlg.belzlg_date, Abs(gs.belzlg_betrag), zlg.belzlg_zahlart,
           lang_text(13521) || ': ' || TBeleg.GetDokumentNr(gs.belzlg_dokument_id), --Automatische Ausgleichszahlung für Gutschrift
           gsID, zlg.belzlg_zl_nr) RETURNING belzlg_id INTO vID;

  -- (Verminderte) Teilzahlung auf Verrechnungsteilzahlung referenzieren lassen.
  UPDATE eingrechzahlung SET
    belzlg_belzlg_id = vID,
    belzlg_betrag = belzlg_betrag - Abs(gs.belzlg_betrag),
    ---belzlg_txt = COALESCE(belzlg_txt,'') || IfThen(belzlg_txt IS NULL,'',' ') || 'Vermindert um Gutschrift ' || TBeleg.GetDokumentNr(gs.belzlg_dokument_id) || ', Betrag: ' || gs.belzlg_betrag::VARCHAR || ' Euro'
    belzlg_txt = COALESCE(belzlg_txt,'') || IfThen(belzlg_txt IS NULL,'',' ') || Format(lang_text(29221) /*'Vermindert um Gutschrift %, Betrag: % Euro'*/, TBeleg.GetDokumentNr(gs.belzlg_dokument_id), gs.belzlg_betrag::VARCHAR)
   WHERE belzlg_id = zlg.belzlg_id;

  -- In Gutschrift Vermerken, dass die mit diesem Beleg gegengerechnet wurde.
  UPDATE eingrechzahlung SET
    belzlg_txt = lang_text(13522) || ' ' || TBeleg.GetDokumentNr(zlg.belzlg_dokument_id) -- Verrechnet mit Beleg-Nr.:
  WHERE belzlg_id = gs.belzlg_id;

  -- Da gibt es schon eine SEPA - Überweisung, die
  IF (zlg.belzlg_sepa_id IS NOT NULL) THEN
    SELECT sepa_seb_id INTO sebid FROM sepaTransaktion WHERE sepa_id = zlg.belzlg_sepa_id;
    DELETE FROM sepaTransaktion WHERE sepa_id = zlg.belzlg_sepa_id;
    zlg :=  eingrechzahlung FROM eingrechzahlung WHERE belzlg_id = zlgid; -- Neu laden, damit Referenzierte Zahlung gesetzt ist.
    PERFORM TBeleg.Create_SepaTransaktion(sebid,zlg);
  END IF;

  RETURN NULL;
 END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- SEPA Kopf-Datensatz für Zahlungslauf anlegen
CREATE OR REPLACE FUNCTION TBeleg.Create_SepaBatch(zlnr VARCHAR(40)) RETURNS INTEGER AS $$
 DECLARE sebid INTEGER;
         zl    Zahlungslauf;
 BEGIN

  zl := zahlungslauf FROM zahlungslauf WHERE zahlungslauf.zl_nr = zlnr;
  IF zl IS NULL THEN
    RAISE EXCEPTION 'TBeleg.Create_SepaBatch: Zahlungslauf % wurde nicht gefunden. Vorgang abgebrochen.', zlnr;
  END IF;

  IF (zl.zl_typ <> 5) THEN
    RAISE EXCEPTION 'TBeleg.Create_SepaBatch: Zahlungslauf % hat einen Typ, der keine SEPA-Überweisungen unterstützt.', zlnr;
  END IF;

  DELETE FROM sepaBatch WHERE seb_zl_nr = zlnr;
  INSERT INTO sepaBatch (seb_zl_nr, seb_kto_name, seb_art) VALUES (zlnr, zl.zl_konto, 0) RETURNING seb_id INTO sebid;

  RETURN sebID;
 END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- Funktion mit Übergabe eines ID Arrays, damit Sortierung beibehalten werden kann.
CREATE OR REPLACE FUNCTION TBeleg.Create_SepaTransaktion(IN sebid INTEGER, IN bzIDs INTEGER[], OUT msg VARCHAR) RETURNS SETOF VARCHAR AS $$
 DECLARE zid INTEGER;
 BEGIN

   FOR zid IN (SELECT belzlg_id
                  FROM eingrechzahlung as e
                  JOIN( SELECT id, row_number() OVER () as sort_id      -- Arraywerte auspacken und nummerieren
                          FROM (SELECT unnest(bzIDs) as id) as ids
                       ) as sorted_ids ON sorted_ids.id = belzlg_id     -- Als Grundlage für JOIN auf Zahlungs-IDs
                ORDER BY sorted_ids.sort_id) LOOP                      -- Nach der wir hinterher sortieren können und Datensätze in der Reihenfolge wie im Array bekommen.
      msg:= TBeleg.Create_SepaTransaktion(sebid, eingrechzahlung) FROM eingrechzahlung WHERE belzlg_id = zid;
     RETURN NEXT;
   END LOOP;
   RETURN;
 END$$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- SEPA Überweisung auf vorbereitete Eingangsrechnungs-Zahlung erstellen
CREATE OR REPLACE FUNCTION TBeleg.Create_SepaTransaktion(sebid INTEGER, bz eingrechzahlung) RETURNS VARCHAR AS $$
 DECLARE msg        VARCHAR;
         r          RECORD;
         sepaid     INTEGER;
         BIC        VARCHAR;
         IBAN       VARCHAR;
         vZweck     VARCHAR;
         refdok     RECORD;
 BEGIN

  msg := '';

  --
  SELECT  beld_id, beld_krzrechnung, beld_refbeleg, adressename(beld_krzrechnung) as adname, ad_landiso,
          a2_krz, a2_knr, a2_eknr, a2_iban, a2_bic, a2_bank, TAdk.Supports_Sepa(ad_landiso) as SepaSupport,
          beld_skv, zl_skontobis, beld_erstelldatum, zl_vorskonto, beld_skonto
          INTO r
     FROM eingrechzahlung
       JOIN belegdokument ON beld_id = belzlg_dokument_id
       JOIN adk           ON beld_krzrechnung = ad_krz
       JOIN adk2          ON a2_krz = ad_krz
       JOIN zahlungslauf  ON belzlg_zl_nr = zl_nr
     WHERE belzlg_id = bz.belzlg_id;

  -- Filter Fehleingaben oder 'Gutschriften' raus.
  IF (COALESCE(bz.belzlg_betrag,0) <= 0) THEN
    msg:= lang_text(29222); --'Betrag muss größer Null sein.';
  END IF;

  --IF (COALESCE(bz.belzlg_date, Current_date)  <  Current_date  )     THEN
    --msg:= msg || IfThen(msg <> '', E'\r\n','') || 'Ausführungsdatum liegt in der Vergangenheit.'; END IF;

  IF (r.a2_krz IS NULL                 ) THEN
    msg:= msg || IfThen(msg <> '', E'\r\n','') || lang_text(29223);  --'Keine Kreditorendaten gefunden.' ;
  END IF;

  IBAN:=TRIM(COALESCE(r.a2_iban,''));
  IF (IBAN = ''  ) THEN
    msg:= msg || IfThen(msg <> '', E'\r\n','') || lang_text(29224);  --'Keine IBAN angegeben.' ;
  END IF;

  BIC:=TRIM(COALESCE(r.a2_bic ,''));  --BIC erforderlich, laut SEPA-Spec für Inland nicht nötig, Bankprogram (ProfiCash) nörgelt aber
  IF (BIC = ''  ) THEN
    msg:= msg || IfThen(msg <> '', E'\r\n','') || lang_text(29225);  --'Keine BIC angegeben.' ;
  END IF;

  IF (TRIM(COALESCE(r.beld_refbeleg,''))='') THEN
    msg:= msg || IfThen(msg <> '', E'\r\n','') || lang_text(29226);  --'Referenz-Belegnummer ist leer.' ;
  END IF;

  IF (NOT r.SepaSupport                ) THEN
     msg:= msg || IfThen(msg <> '', E'\r\n','') || lang_text(29227);  --'Empfängerland ist kein SEPA-Mitglied.' ;
  END IF;

  IF (msg<>'') THEN
    msg := lang_text(29228) /*'Fehler beim Erstellen der SEPA-Überweisung. (Lieferant, RG-Nr:*/ || ' '
            || COALESCE(r.beld_krzrechnung,'?') || ', ' || COALESCE(r.beld_refbeleg,'?')|| ')'
            || E'\r\n' || msg ;
    RETURN msg;
  END IF;

  sepaID := belzlg_sepa_id FROM eingrechzahlung WHERE belzlg_id = bz.belzlg_id;
  IF SepaID IS NOT NULL THEN
    DELETE FROM sepaTransaktion WHERE sepa_id = sepaID;
  END IF;

  vZweck := 'RG: '||r.beld_refbeleg;
  IF NULLIF(TRIM(r.a2_eknr),'') IS NOT NULL THEN
    vZweck := vZweck || ', Kd.-Nr. ' || TRIM(r.a2_eknr);
  END IF;

  -- Skontoprüfung, Skontosatz vorhanden und im Zeitraum?
  IF (COALESCE( r.beld_skv, 0) > 0) AND ((r.beld_erstelldatum::Date) + r.beld_skv) >= bz.belzlg_date THEN
    --vZweck := vZweck || ', abz. Skonto ' || r.beld_skonto::NUMERIC(12,2) || ' Euro';
    vZweck := Format(lang_text(29229) /*'%, abz. Skonto % Euro'*/, vZweck, r.beld_skonto::NUMERIC(12,2));
  END IF;

  -- Gutschriftenverrechnung: 'Richtige Zahlung' -Ref-> 'Verrechneter Betrag' -Ref-> Gutschriftszahlung (siehe auch: TBeleg.Assign_Gutschrift)
  IF bz.belzlg_belzlg_id IS NOT NULL THEN -- Aktuelle Teilzahlung zeigt auf andere Teilzahlung
    -- Dokument der referenzierten Teilzahlung holen
    SELECT beld_id, beld_dokunr, beld_refbeleg, eingrechzahlung.* INTO refDok
      FROM eingrechzahlung
        JOIN eingrechdokument ON belzlg_dokument_id = beld_id
      WHERE belzlg_id = bz.belzlg_belzlg_id;

    -- Referenzierte Zahlung hängt an einem anderen Dokument, wie behandelte. => Geht nur, wenn wir selbst die Verrechnungs-Teilzahlung sind.
    IF (refdok.beld_id <> r.beld_id) THEN
      RETURN lang_text(29230);  --'Verrechnungsposition, wird übersprungen';
    END IF;
  END IF;

  -- Referenzierte Teilzahlung ist auf gleichem Dokument -> Aktuelle Teilzahlung wurde um  Gutschrift gekürzt.
  IF (bz.belzlg_belzlg_id IS NOT NULL) THEN
    IF (refdok.beld_id IS NOT DISTINCT FROM r.beld_id) THEN
      --vZweck := COALESCE(vZweck,'') || IfThen(COALESCE(vzweck,'')='','',', ')
      --       || 'abz. GS ' || refdok.beld_refbeleg || ', ' || refdok.belzlg_betrag || ' Euro';
      vZweck := COALESCE(vZweck,'') || IfThen(COALESCE(vzweck,'')='','',', ') || Format(lang_text(29231) /*'abz. GS %, % Euro'*/, refdok.beld_refbeleg, refdok.belzlg_betrag);
    END IF;
  END IF;

  INSERT INTO sepaTransaktion ( sepa_seb_id, sepa_adkrz, sepa_name, sepa_iban,
                                sepa_bic, sepa_bank, sepa_endToEndID, sepa_vzweck,
                                sepa_betrag, sepa_termin, sepa_datum )
    SELECT sebid, r.beld_krzrechnung, substring(COALESCE(r.adname,'') FROM 0 FOR 27), IBAN,
            BIC, r.a2_bank, r.beld_refbeleg, vZweck,
            bz.belzlg_betrag,
            IfThen( bz.belzlg_date = current_date, NULL, bz.belzlg_date), --Heute => Sofort überweisen, Datum leer lassen.
            bz.belzlg_date
    RETURNING sepa_id INTO sepaid;

  UPDATE eingrechzahlung SET belzlg_sepa_id = sepaid WHERE belzlg_id = bz.belzlg_id;

  RETURN msg;
 END $$ LANGUAGE plpgsql VOLATILE STRICT;
--

-- Belegzahlung verbuchen
CREATE OR REPLACE FUNCTION TBeleg.Finish_ERGZahlung( belzlgid INTEGER, ZahlDatum DATE DEFAULT CURRENT_DATE) RETURNS VARCHAR AS $$
 DECLARE r         RECORD;
         offen     NUMERIC;
         isDone    BOOLEAN;
         skontiert BOOLEAN;
         msg       VARCHAR;
 BEGIN

  SELECT beld_erstelldatum, beld_skv, beld_id, belzlg_betrag, belzlg_zl_nr, beld_dokunr, sepa_id, seb_msgID, beld_bezahlt, belzlg_belzlg_id INTO r
    FROM eingrechdokument
      JOIN eingrechzahlung      ON belzlg_dokument_id = beld_id
      LEFT JOIN sepaTransaktion ON belzlg_sepa_id     = sepa_id
      LEFT JOIN sepaBatch       ON sepa_seb_id        = seb_id
    WHERE belzlg_id = belzlgid;

  -- Prüfen ob wir mit dem geplantem Zahlungsdatum noch im Skontozeitraum sind
  skontiert := ZahlDatum <= (r.beld_erstelldatum + r.beld_skv);

  -- Offene Summe mit oder ohne Skonto
  offen  := TBeleg.DokumentSumme( r.beld_id, FALSE, FALSE, TRUE, skontiert, FALSE) - r.beld_bezahlt;

  -- Zahlbetrag größergleich offenem Betrag?
  isDone := (offen - r.belzlg_betrag) <= 0;

  UPDATE eingrechdokument SET
    beld_verbucht       = isDone,
    beld_zahl_skontiert = skontiert,
    beld_abschlussdatum = Ifthen(isDone, zahldatum, NULL)
  WHERE beld_id = r.beld_id;

  UPDATE eingrechzahlung SET belzlg_gebucht = TRUE, belzlg_tan = r.seb_msgID WHERE belzlg_id = belzlgid;
  IF (r.belzlg_belzlg_id IS NOT NULL) THEN
    PERFORM TBeleg.Finish_ERGZahlung(r.belzlg_belzlg_id,Zahldatum); -- "Verminderte Zahlung" => schließt "Verrechnungszahlung" => Schließt "Gutschriftenposition"
  END IF;

  IF (isDone) THEN
    --msg := 'Zahlungslauf ' || r.belzlg_zl_nr ||': Eingangsrechnung erledigt. Beleg-Nr: ' || r.beld_dokunr;
    msg := Format(lang_text(29232) /*'Zahlungslauf %: Eingangsrechnung erledigt. Beleg-Nr: %'*/, r.belzlg_zl_nr, r.beld_dokunr);
  ELSE
    --msg := 'Zahlungslauf ' || r.belzlg_zl_nr ||': Teilzahlung verbucht. Beleg-Nr: ' || r.beld_dokunr;
    msg := Format(lang_text(29233) /*'Zahlungslauf %: Teilzahlung verbucht. Beleg-Nr: %'*/, r.belzlg_zl_nr, r.beld_dokunr);
  END IF;
  RETURN msg;
 END $$ LANGUAGE plpgsql VOLATILE STRICT;
--


-- [OBSOLET] Altes DTA Format. Seit 08/2014 - Nur zu Vergleichszwecken hier wg. Altdaten bei SUN
/* CREATE TABLE dtaus (
      dta_id                SERIAL PRIMARY KEY,
      dta_zweck             VARCHAR(351),   --Verwendungszweck (Maximal 13 Erweiterungsteile á 27 Zeichen), bei Sunstrom nur 10 Teile wg. Banksoftware.
      dta_empfaenger        VARCHAR(27),
      dta_empf_ktnr         VARCHAR(10),
      dta_empf_blz          VARCHAR(8),
      dta_empf_bank         VARCHAR(60),
      dta_adkrz             VARCHAR(40) REFERENCES adk ON UPDATE CASCADE,
      dta_einzahler         VARCHAR(27),
      dta_einz_ktnr         VARCHAR(10),
      dta_einz_blz          VARCHAR(8),
      dta_einz_bank         VARCHAR(60),
      dta_betrag            NUMERIC(12,2) NOT NULL DEFAULT 0,
      dta_ausfuehrdat       DATE,
      dta_zl_nr             VARCHAR(40) REFERENCES zahlungslauf ON UPDATE CASCADE ON DELETE CASCADE,
      dta_valid             BOOLEAN NOT NULL DEFAULT TRUE
     );
    --

     -- Validierung
     CREATE OR REPLACE FUNCTION dtaus__b_iu() RETURNS TRIGGER AS $$
      BEGIN
      --Wir gehen erstmal von Validen Kontendaten aus, setzen dann FALSE wenn was nicht stimmt.
      new.dta_valid:=TRUE;

      --Alle benötigten Daten da?
      IF ( (new.dta_empfaenger IS NULL) OR (new.dta_empf_ktnr IS NULL) OR (new.dta_empf_ktnr IS NULL) OR (new.dta_empf_blz IS NULL)
                OR (new.dta_einzahler IS NULL) OR (new.dta_einz_ktnr IS NULL) OR (new.dta_einz_blz IS NULL)) THEN
        new.dta_valid :=FALSE;
      END IF;

      --Null-Beträge?
      IF (new.dta_betrag <= 0 ) THEN
        new.dta_valid := FALSE;
      END IF;

      --BLZ richtige Länge?
      IF ( ( new.dta_empf_blz > '99999999') OR (new.dta_empf_blz < '1000000') OR ( new.dta_einz_blz > '99999999') OR (new.dta_einz_blz < '1000000') ) THEN
        new.dta_valid :=FALSE;
      END IF;

      RETURN new;
      END $$ LANGUAGE plpgsql;
      --
     CREATE TRIGGER dtaus__b_iu
      BEFORE INSERT OR UPDATE
      ON dtaus
      FOR EACH ROW
      EXECUTE PROCEDURE dtaus__b_iu();
      --

     --
     CREATE OR REPLACE FUNCTION dtaus__b_d() RETURNS TRIGGER AS $$
      BEGIN
      UPDATE eingrechzahlung SET belzlg_dta_id = NULL WHERE belzlg_dta_id = old.dta_id;
      RETURN old;
     END $$ LANGUAGE plpgsql;
     --
     CREATE TRIGGER dtaus__b_d
      BEFORE DELETE
      ON dtaus
      FOR EACH ROW
      EXECUTE PROCEDURE dtaus__b_d();
      --
    --

*/

/************************************************************
PREISE BUCHEN, Zurückschreiben aus Eingangsrechnungen in Artikelstamm, Lieferantendaten und Nachkalkulation
*************************************************************/

--
CREATE OR REPLACE FUNCTION TBeleg.BelPreisBuchen(
      beldokid      integer,
      avgprice      boolean,
      nachkalk      boolean,
      selbstko      boolean,
      vkpbasis      boolean,
      doepreis      boolean,

      OUT pos       varchar,
      OUT aknr      varchar,
      OUT strAVG    varchar,
      OUT strSK     varchar,
      OUT strVKP    varchar,
      OUT strEPreis varchar,
      OUT strNK     varchar
  ) RETURNS SETOF record AS $$
  DECLARE
      _rec          record;
      _eprec        record;
      _preis        numeric(12,4);
      _selbstkosten numeric(12,4);
      _vkp          numeric(12,4);
      _eid          integer;
      _sk_alt       numeric(12,4);
      _vkp_alt      numeric(12,4);
      _posabzunetto numeric(12,4);

      -- Systemeinstellung: Beim Verbuchen sollen Rechnungen geprüft gesetzt werden.
      _DoGeprueft   boolean := TSystem.Settings__GetBool('ERBuchenSetztGeprueft');
  BEGIN
      -- Preise verbuchen:
       -- zurückschreiben aus Eingangsrechnungen in Artikelstamm, Lieferantendaten und Nachkalkulation


      -- Erledigte Positionen sind schon gebucht, oder sollen übersprungen werden.
      FOR _rec IN
          SELECT
            -- ERG-Belegnummer
            beld_dokunr,
            -- Position der Eingangsrechnung
            eingrech_pos.*,
            -- Daten zum Fertigungsartikel bei ERG zum ABK-Arbeitsgang (Auswärts-ERG)
            (tabk.abk__fertdata__by__a2_id( belp_a2_id )).*,
            -- Selbskosten bei Verbuchung
            coalesce( lo_hest, ak_hest, 0 ) AS lo_ak_hest,
            -- Lagernde Menge bei Verbuchung abzgl. zugebuchter Menge aus der ERG-Position
            numeric_larger( round( coalesce( lo_bestand, ak_tot ) - belp_menge_gme ), 0 ) AS lagernd,
            -- Artikeldatena
            ak_tot, ak_vkpfaktor, ak_fertigung, ak_vkpbas,
            -- AC-Daten
            ac_n, ac_vkpfaktor,
            -- Bestellung
            ld_ag_id, ld_abk,
            -- Lagerbewegungs-LOG des Wareneingangs
            lo_id,
            -- #15153 keine Übernahme in Selbstkosten aus QS
            (ld_auftg LIKE '%.QS' OR ld_q_nr IS NOT NULL) AS qs_auftg,
            -- #13968 letzer berechneter Durchschnittpreis vor Buchung
            TLager.lagerlog__lo_adpreis__get_last_before_timestamp(ak_nr, lo_datum) as last_lo_adpreis
          FROM   eingrech_pos
            JOIN eingrechdokument ON belp_dokument_id = beld_id
            JOIN art              ON belp_aknr = ak_nr
            JOIN artcod           ON ak_ac = ac_n
            LEFT JOIN ldsdok      ON belp_ld_id = ld_id
            LEFT JOIN lagerlog    ON lo_wendat = belp_w_wen AND lo_aktion = '+'
          WHERE belp_dokument_id = beldokid
            AND belp_belegtyp = 'ERG'
            AND NOT belp_erledigt
      LOOP

          -- Ausgabe
          pos       := _rec.beld_dokunr || ' / ' || _rec.belp_pos;
          aknr      := _rec.belp_aknr;

          _sk_alt   := _rec.lo_ak_hest;
          _vkp_alt  := _rec.ak_vkpbas;

          -- Durchschnitts-EK-Preis für Lager (wird seit 2013-01-07 immer gemacht, Einstellung 'avgprice' entfernt)
          -- Nur wenn Lagerzugang vorhanden und das kein Fertigungsartikel ist
          -- (Ab 09/2014) nur, wenn das kein ERG auf Auswärtsbearbeitung ist, Durchschnitts-EK über ein Arbeitspaket
          -- ist relativ sinnlos.
          -- zumal es zwar den Lagerzugang gibt, aber keinen Lagerbestand.
          IF
              -- Wareneingang für ERG vorhanden
                  _rec.belp_w_wen IS NOT NULL
              -- und kein Fertigungsartikel
              AND NOT _rec.ak_fertigung
              -- und kein Bezug zu ABK-AG (kein Auswärts-ERG)
              AND _rec.belp_a2_id IS NULL
              -- Beistellung zerlegt mit abweichenden Werten die Daten
              AND _rec.ld_abk IS NULL
          THEN

              --belp_netto ist Pos-Gesamtpreis auf gesamte Stückzahl mit Abzu/Rabatt etc ...
              _preis :=   (   coalesce( _rec.last_lo_adpreis, _rec.lo_ak_hest ) * _rec.lagernd  -- Bestandswert: letzer Durchschnittspreis falls vorhanden, sonst SK
                            + _rec.belp_netto_basis_w                                           -- Positionswert ERG
                          )
                        /
                          do1if0( _rec.lagernd + _rec.belp_menge_gme );                         -- gesamte Lagermenge

              -- auf 2 Stellen runden
              _preis := _preis::numeric(19,2);

              -- Preise zurückschreiben
              PERFORM disablemodified();

              -- in Wareneingang
              UPDATE wendat SET
                w_adpreis_berech = true,
                w_adpreis = _preis
              WHERE w_wen = _rec.belp_w_wen
                AND NOT w_adpreis_berech
              ;

              -- in Lagerbewegungs-LOG des Wareneingangs
              UPDATE lagerlog SET
                lo_adpreis = _preis
              WHERE lo_id = _rec.lo_id
              ;

              PERFORM enablemodified();

              -- Ausgabe
              strAVG := _rec.lo_ak_hest::numeric(12,2) || ' => ' || _preis::numeric(12,2);

          END IF;


          -- Preise vorberechnen, falls wir in Selbstkosten, VKP-Basis oder Nachkalkulation zurückschreiben
          IF
                  selbstko
              OR  vkpbasis
              OR  nachkalk
          THEN

              -- Abzüge und Zuschläge der Position
              _posabzunetto :=
                    sum( coalesce( belpaz_betrag, 0 ) * belpaz_anzahl )
                  FROM belegposabzu
                    JOIN abzu ON belpaz_abzu_id = abz_id
                  WHERE belpaz_belegpos_id = _rec.belp_id
                  AND abz_inselbstko
              ;

              _posabzunetto := coalesce( _posabzunetto, 0 );

              --Abzüge und Zuschläge berücksichtigen, wenn abz_inSelbstKo gesetzt.
              _selbstkosten := _rec.belp_preis_gme_basis_w * IfThen( _rec.belp_rabattfaehig, ( 1 - _rec.belp_rabatt / 100 ), 1 );
              _selbstkosten := _selbstkosten + ( _posabzunetto  / Do1If0( _rec.belp_menge_gme ) );

              -- Verkaufspreis-Basis
              _vkp := _selbstkosten * coalesce( _rec.ak_vkpfaktor, _rec.ac_vkpfaktor );

          END IF;


          -- Selbstkosten / VKP-Basis, wenn kein Arbeitspaket
          -- Selbstkosten berücksichtigen Rabatte und "selbstkostenfaehige" Ab- und Zuschläge
          IF
              -- Selbskosten oder VKP-Basis soll berechnet werden
                  ( selbstko OR vkpbasis )
              -- und kein Bezug zu ABK-AG (kein Auswärts-ERG)
              AND _rec.belp_a2_id IS NULL
              -- Beistellung zerlegt mit abweichenden Werten die Daten
              AND _rec.ld_abk IS NULL
              -- QS ausschließen, siehe #15153
              AND NOT _rec.qs_auftg
          THEN

              -- Preise zurückschreiben
              PERFORM disablemodified();

              UPDATE art SET
                -- Selbstkosten übernehmen oder belassen
                ak_hest   = IfThen( selbstko, _selbstkosten, ak_hest ),
                -- VKP-Basis übernehmen oder belassen
                ak_vkpbas = IfThen( vkpbasis, _selbstkosten * ( coalesce( ak_vkpfaktor, _rec.ac_vkpfaktor, 1 ) ), ak_vkpbas )
              WHERE ak_nr = _rec.belp_aknr
                AND NOT ak_fertigung
              ;

              -- Wenn UPDATE erfolgt ist, Herkunft und Ausgaben aktualisieren
                -- #6164 und ain_hest_datum wird in art__a_iu() gesetzt
              IF FOUND THEN

                  -- Preisherkunft in Artikelstamm aktualisieren
                  PERFORM TArtikel.ain_hest_herkunft__set( aknr, 'erg', pos );

                  IF selbstko THEN
                      -- Ausgabe
                      strSK := _sk_alt || ' => ' || _selbstkosten;
                  END IF;

                  IF vkpbasis THEN
                      -- Ausgabe
                      strVKP := _vkp_alt  || ' => ' || _vkp;
                  END IF;

              END IF;

              PERFORM enablemodified();

          END IF;


          --In Lieferantendaten zurückschreiben
          IF doepreis THEN

              -- Für das Rechnungsdatum und den Artikel schon ein Satz bei diesem Lieferanten? Dann updaten, sonst neu anlegen.
              SELECT *
              FROM epreis
              WHERE
                -- Lieferant stimmt überein
                    e_lkn = _rec.belp_krzlieferung
                -- Artikel stimmt überein
                AND e_aknr = _rec.belp_aknr
                -- Falls Fertigungsartikel-Nr. dann muss die auch stimmen,
                -- damit man gleiches AP für unterschiedliche Fert.Artikel erfassen kann.
                AND coalesce( e_fertaknr, '' ) = coalesce( _rec.fertaknr, '' )
                -- gleiches gültig ab Datum
                AND e_gdatum = _rec.belp_erstelldatum
              ORDER BY e_gdatum, e_id
              LIMIT 1
              INTO _eprec
              ;


              -- Ohne Lieferantepreis-Eintrag neuen erzeugen
              IF _eprec.e_id IS NULL THEN

                  INSERT INTO epreis (
                      e_aknr,                   e_fertaknr,
                      e_lkn,                    e_best,
                      e_preis,                  e_preiseinheit,
                      e_waer,                   e_kurs,
                      e_rab,                    e_stk,            e_mcv,
                      e_lfzt,
                      e_gdatum,                 e_herkunft
                    )
                  VALUES (
                      _rec.belp_aknr,           _rec.fertaknr,
                      _rec.belp_krzlieferung,   _rec.belp_referenzaknr::varchar(40),
                      _rec.belp_preis,          Do1If0( _rec.belp_preiseinheit ),
                      _rec.belp_waehr,          _rec.belp_kurs,
                      _rec.belp_rabatt,         _rec.belp_menge,  _rec.belp_mce,
                      -- Laufzeit ermitteln
                      coalesce(
                          (
                              SELECT e_lfzt
                              FROM epreis
                              WHERE e_aknr  = _rec.belp_aknr
                                AND e_lkn   = _rec.belp_krzlieferung
                              ORDER BY e_id DESC
                              LIMIT 1
                          )
                          , 0
                      ),

                      _rec.belp_erstelldatum,   'ER = ' || coalesce( pos, '' )
                    )
                  ;

              -- Lieferantepreis-Eintrag aktualisieren
              ELSE

                  PERFORM disablemodified();

                  UPDATE epreis SET
                    e_ep        = _rec.belp_preis / Do1If0( _rec.belp_preiseinheit ),
                    e_waer      = _rec.belp_waehr,
                    e_rab       = _rec.belp_rabatt,
                    e_stk       = _rec.belp_menge,
                    e_mcv       = _rec.belp_mce,
                    e_herkunft  = 'ER = ' || coalesce( pos, '' ),
                    e_kurs      = _rec.belp_kurs
                  WHERE e_id = _eprec.e_id
                  ;

                  PERFORM enablemodified();

              END IF;


              --E_id holen, die wir geupdated oder neu angelegt haben
              _eid :=
                    e_id
                  FROM epreis
                  WHERE e_aknr = _rec.belp_aknr
                    AND e_lkn = _rec.belp_krzlieferung
                    AND e_gdatum = _rec.belp_erstelldatum
                  ORDER BY e_id DESC
                  LIMIT 1
              ;

              -- Abzüge und Zuschläge löschen und neu übernehmen
              DELETE FROM epreisabzu WHERE eaz_e_id = _eid;

              -- Position-Abzüge und -Zuschläge aus ERG
              INSERT INTO epreisabzu
                ( eaz_e_id, eaz_abz_id,     eaz_anz,        eaz_betr,       eaz_proz,       eaz_canskonto )
              SELECT
                  eid,      belpaz_abzu_id, belpaz_anzahl,  belpaz_betrag,  belpaz_prozent, belpaz_canskonto
              FROM belegposabzu
              WHERE belpaz_belegpos_id = _rec.belp_id
              ;

              -- Belege-Abzüge und -Zuschläge aus ERG in Summe
              IF coalesce( _rec.belp_sumBelegAbzu, 0 ) <> 0 THEN

                  INSERT INTO epreisabzu
                    ( eaz_e_id, eaz_abz_id, eaz_anz,  eaz_betr,               eaz_proz, eaz_canskonto )
                  VALUES
                    ( _eid,     200,        1,        _rec.belp_sumBelegAbzu, 0,        false )
                  ;

              END IF;


              -- Ausgabe
              strEPreis := _rec.belp_preis::numeric(12,2);

          END IF;


          -- In Nachkalkulationspreis zurückschreiben
          IF nachkalk THEN

              PERFORM disableauftgtrigger();
              -- Der Preis der Eingangsrechnung geht in die Nachkalkulationen! Durch den Trigger "auftg__a_iu__nkchange"
              -- wird abk.ab_nk_change auf true gesetzt.

              -- Aktualisierung der Materialpositionen, die anhand Buchungen diese ERG verwendet haben.
                -- Durchschnittspreis aus versch. ABK, Bestellungen, ERG neu berechnen.
                -- löst ab_nk_change auf true aus (auftg__a_iu__nkchange)
              PERFORM tplanterm.auftg_nk_calc_vkp( null, agids )
              -- LA kommt aus Bestellung und wird für MatPos verwendet. Bestellung ist für unsere ERG.
              FROM (
                  SELECT DISTINCT l_ag_id AS agids
                  FROM lifsch
                  WHERE l_ag_id IS NOT NULL
                    AND l_ld_id = _rec.belp_ld_id
              ) AS sub
              ;

              -- Fallback: direkte MatPos aus Bestellung mit BelPreis füllen, wenn kein Durchschnittspreis erstellt wurde.
              UPDATE auftg SET
                ag_nk_calc_vkp_uf1 = _selbstkosten
              WHERE ag_id = _rec.ld_ag_id
                AND ag_nk_calc_vkp_uf1 IS NULL
              ;

              PERFORM enableauftgtrigger();

              -- Ausgabe
              strNK := _selbstkosten;

          END IF;

          -- Debugging
          RAISE NOTICE 'Pos Done: %', _rec.belp_pos;

          RETURN NEXT;

      END LOOP;


      -- Verbuchung abschließen
        -- Trigger deaktivieren. Bei Sammelrechnungen läuft PicnDoku Verschlagwortung sonst in Timeout.
        -- Die wird hier aber auch nicht gebraucht, da keine Schlagworte geändert werden.
      PERFORM DisableBelPosTrigger();

      -- ERG-Positionen abschließen
      UPDATE eingrech_pos SET
        belp_erledigt = true
      WHERE belp_dokument_id = beldokid
        AND NOT belp_erledigt
      ;

      PERFORM EnableBelPosTrigger();

      -- ERG-Belege definitiv setzen.
        -- In Oberflaeche wird gewarnt, wenn def. Beleg geaendert werden soll.
      UPDATE eingrechdokument SET
        beld_definitiv = true,
        beld_geprueft = IfThen( _DoGeprueft AND beld_pruefer IS NULL, true, beld_geprueft )
      WHERE beld_id = beldokid
      ;


  END $$ LANGUAGE plpgsql;
--



/***************************************************************************
HILFSFUNKTIONEN / SONSTIGES
*************************************************************/




-- Zahlungslauf wieder zur Bearbeitung öffnen, auch wenn der schon gebucht war.
CREATE OR REPLACE FUNCTION TEingRech.OpenZahlungslauf(zlnr VARCHAR) RETURNS VOID AS $$
 BEGIN

  UPDATE zahlungslauf SET zl_vorbereitet=FALSE, zl_gebucht = FALSE
        WHERE zl_nr = zlnr;

  UPDATE eingrechdokument SET beld_zahl_skontiert = FALSE, beld_verbucht=FALSE, beld_abschlussdatum = NULL
        FROM eingrechzahlung
        WHERE
        belzlg_zl_nr IS NOT NULL
        AND belzlg_dokument_id = beld_id
        AND belzlg_zl_nr = zlnr;

  UPDATE eingrechzahlung SET belzlg_gebucht=FALSE WHERE belzlg_zl_nr = zlnr;

  UPDATE sepaTransaktion SET sepa_done = FALSE
    FROM sepaBatch
    WHERE seb_zl_nr IS NOT NULL
      AND seb_zl_nr = zlnr
      AND sepa_seb_id = seb_id
      AND sepa_done;

 END $$ LANGUAGE 'plpgsql';
--


CREATE OR REPLACE FUNCTION TEingRech.GetMaxMahnstufe(a2krz VARCHAR) RETURNS INTEGER AS $$
 DECLARE mStufe INTEGER;
 BEGIN
  SELECT MAX(beld_mahnstufe) INTO mStufe
        FROM eingrechdokument
        WHERE   beld_mahnstufe IS NOT NULL
                AND beld_krzrechnung = a2krz
                AND beld_verbucht    = FALSE;
  RETURN  COALESCE(mStufe,0);
 END $$ LANGUAGE plpgsql STABLE;


--Preis aus letzter erfasster Eingangsrechnung (pro angegebene ME)
CREATE OR REPLACE FUNCTION TEingRech.letzte_einkaufsPreis(aknr VARCHAR, artmgcid INTEGER, addBelAbzu BOOLEAN = FALSE, get_beistellung BOOLEAN=TRUE) RETURNS NUMERIC AS $$
 BEGIN
  RETURN TEingRech.letzte_einkaufsPreis_dat(aknr, artmgcid, current_date, addBelAbzu, get_beistellung);
 END $$ LANGUAGE plpgsql;

--+COALESCE((SELECT SUM(belpaz_betrag*belpaz_anzahl) FROM belegposabzu JOIN abzu ON abz_id=belpaz_abzu_id WHERE belpaz_belegpos_id=belp_id AND abz_inselbstko),0)/(IfThen(belp_menge_gme=0, 1, belp_menge_gme))

--Preis aus letzter erfasster Eingangsrechnung (pro angegebene ME), inkl. aller Positionsabzuschläge
CREATE OR REPLACE FUNCTION TEingRech.letzte_einkaufsPreis_dat(aknr VARCHAR, artmgcid INTEGER, dat DATE, addBelAbzu BOOLEAN = FALSE, get_beistellung BOOLEAN=TRUE) RETURNS NUMERIC AS $$
 DECLARE preis           NUMERIC;
         preisBelAbzu    NUMERIC;
 BEGIN
    IF ak_fertigung FROM art WHERE ak_nr = aknr THEN -- eigentlich Fertigungsartikel, dann kein EingangsrechnungsPreis!
       RETURN NULL;
    END IF;

    --Pos.Preis mit Pos.Abzu / Menge ... umgerechnet in gewuenschte Mengeneinheit
    SELECT
           coalesce(IfThen(belp_menge_gme = 0, 0, belp_netto_basis_w / Do1If0(belp_menge_gme) ), 0 ),
           belp_abzupreis_gme
      INTO preis,preisBelAbzu
      FROM belegpos
      LEFT JOIN ldsdok ON ld_id = belp_ld_id
     WHERE belp_aknr = aknr
       AND belp_netto > 0
       AND belp_erstelldatum <= dat
       AND CASE WHEN get_beistellung THEN
                     TRUE
                ELSE ld_abk IS NULL
                END
     ORDER BY belp_erstelldatum DESC LIMIT 1;

    IF addBelAbzu THEN
       preis := preisBelabzu;
    END IF;

    RETURN tartikel.me__menge__in__menge_uf1(artmgcid, preis);
END $$ LANGUAGE plpgsql;

--Rechnungsnummer aus letzter erfasster Eingangsrechnung (pro angegebene ME), inkl. aller Positionsabzuschläge
CREATE OR REPLACE FUNCTION TEingRech.letzte_einkaufsRechNr(aknr VARCHAR, get_beistellung BOOLEAN=TRUE) RETURNS VARCHAR AS $$
 BEGIN
  RETURN beld_dokunr
  FROM belegpos
    JOIN belegdokument ON belp_dokument_id = beld_id
    LEFT JOIN ldsdok ON ld_id=belp_ld_id
    WHERE belp_aknr = aknr AND belp_netto>0 AND belp_erstelldatum<=current_date AND CASE WHEN get_beistellung THEN TRUE ELSE ld_abk IS NULL END ORDER BY belp_erstelldatum DESC LIMIT 1;
 END $$ LANGUAGE plpgsql;

--Alle Belege und Pos. auf denen ein Wareneingang auftaucht.
CREATE OR REPLACE FUNCTION TEingRech.BelegPosZuWendat(wwen INTEGER) RETURNS VARCHAR AS $$
 DECLARE ret VARCHAR;
        rec RECORD;
 BEGIN
  FOR rec IN (SELECT beld_dokunr, belp_pos FROM eingrechdokument JOIN belegpos ON belp_dokument_id = beld_id WHERE belp_w_wen = wwen) LOOP
    IF (COALESCE(ret,'') <> '') THEN
      ret:=ret||', ';
    END IF;
    ret:=COALESCE(ret,'') || rec.beld_dokunr || '/' || rec.belp_pos;
  END LOOP;

  RETURN COALESCE(ret,'');
 END $$ LANGUAGE 'plpgsql';


-- keine leeren Statements am Ende vom Erstellen der DB erlaubt.
SELECT TRUE;
